<?php
/**
 * Plugin Name: Easy Digital Downloads - Social Login
 * Plugin URI: https://wpwebelite.com
 * Description: Allow your customers to login and checkout with social networks such as Facebook, Twitter, Google, Yahoo, LinkedIn, Foursquare, Windows Live, VK.com Amazon, PayPal and Apple.
 * Version: 2.5.6
 * Author: WPWeb
 * Author URI: https://wpwebelite.com
 * Text Domain: eddslg
 * Domain Path: languages
 * Tested up to: 6.3.1
 * 
 * @package Easy Digital Downloads - Social Login
 * @category Core
 * @author WPWeb
 */
// Exit if accessed directly
if( !defined('ABSPATH') ) exit;

/**
 * Basic plugin definitions 
 * 
 * @package Easy Digital Downloads - Social Login
 * @since 1.0.0
 */
if( !defined('EDD_SLG_VERSION') ) {
	define( 'EDD_SLG_VERSION', '2.5.6' ); // Plugin version
}
if( !defined('EDD_SLG_URL') ) {
	define( 'EDD_SLG_URL', plugin_dir_url(__FILE__) ); // plugin url
}
if( !defined('EDD_SLG_DIR') ) {
	define( 'EDD_SLG_DIR', dirname(__FILE__) ); // plugin dir
}
if( !defined('EDD_SLG_SOCIAL_DIR') ) {
	define( 'EDD_SLG_SOCIAL_DIR', EDD_SLG_DIR . '/includes/social' ); // social dir
}
if( !defined('EDD_SLG_SOCIAL_LIB_DIR') ) {
	define( 'EDD_SLG_SOCIAL_LIB_DIR', EDD_SLG_DIR . '/includes/social/libraries' ); // lib dir
}
if( !defined('EDD_SLG_IMG_URL') ) {
	define( 'EDD_SLG_IMG_URL', EDD_SLG_URL . 'includes/images' ); // image url
}
if( !defined('EDD_SLG_ADMIN') ) {
	define( 'EDD_SLG_ADMIN', EDD_SLG_DIR . '/includes/admin' ); // plugin admin dir
}
if( !defined('EDD_SLG_USER_PREFIX') ) {
	define( 'EDD_SLG_USER_PREFIX', 'edd_user_' ); // username prefix
}
if( !defined('EDD_SLG_USER_META_PREFIX') ) {
	define( 'EDD_SLG_USER_META_PREFIX', 'eddslg_' ); // username prefix
}
if( !defined('EDD_SLG_BASENAME') ) {
	define( 'EDD_SLG_BASENAME', 'edd-social-login' );
}

$upload_dir = wp_upload_dir();
$upload_path = isset($upload_dir['basedir']) ? $upload_dir['basedir'] . '/' : ABSPATH;
$upload_url = isset($upload_dir['baseurl']) ? $upload_dir['baseurl'] : site_url();

// edd social login upload folder path
if( !defined('EDD_SLG_TMP_DIR') ) {
	define( 'EDD_SLG_TMP_DIR', $upload_path . 'edd/eddslg-tmp' );
}
if( !defined('EDD_SLG_PLUGIN_KEY') ) {
	define( 'EDD_SLG_PLUGIN_KEY', 'eddslg' );
}
if( !defined('EDD_SLG_SOCIAL_BLOCK_DIR') ) {
	define( 'EDD_SLG_SOCIAL_BLOCK_DIR', EDD_SLG_DIR . '/includes/blocks/' ); // block dir
}

// Required Wpweb updater functions file
if( !function_exists('wpweb_updater_install') ) {
	require_once( 'includes/wpweb-upd-functions.php' );
}

/**
 * Admin notices
 * 
 * @package Easy Digital Downloads - Social Login
 * @since 1.0.0
 */
function edd_slg_admin_notices() {
	if( !class_exists('Easy_Digital_Downloads') ) {
		echo '<div class="error">';
		echo "<p><strong>" . esc_html__( 'Easy Digital Downloads needs to be activated to be able to use the Social Login.', 'eddslg' ) . "</strong></p>";
		echo '</div>';
	}
}

/**
 * Adding Notice for Facebook SDK
 */
function edd_slg_admin_notices_for_facebook_SDK() {
	if( !get_option('edd_slg_fb_sdk_notice_disable') && current_user_can('administrator') ) {
		echo '<div class="notice notice-warning is-dismissible ">'
		. '<a href="?edd_slg_fb_sdk_notice_disable=1" class="edd-slg-facebook-sdk-link">Dismiss</a>'
		. '<p>' . esc_html__('We have upgraded Facebook SDK, So you need to update redirect URL in your Facebook app. Please setup redirect URL to your Facebook app as per new settings called "Facebook Valid OAuth Redirect URL".', 'eddslg') . '</p>'
		. '</div>';
	}
}
add_action( 'admin_notices', 'edd_slg_admin_notices_for_facebook_SDK', 999 );

/**
 * Dismiss the Notice of Facebook
 */
function edd_slg_facebook_admin_notices_dismissed() {
	if( !empty($_GET['edd_slg_fb_sdk_notice_disable']) ) {
		update_option( 'edd_slg_fb_sdk_notice_disable', '1' );
	}
}
add_action('admin_init', 'edd_slg_facebook_admin_notices_dismissed');

/**
 * Check Easy Digital Downloads Plugin
 *
 * Handles to check Easy Digital Downloads plugin
 * if not activated then deactivate our plugin
 *
 * @package Easy Digital Downloads - Social Login
 * @since 1.0.0
 */
function edd_slg_check_activation() {

	if( !class_exists('Easy_Digital_Downloads') ) {
		// is this plugin active?
		if( is_plugin_active(plugin_basename(__FILE__)) ) {
		// deactivate the plugin
			deactivate_plugins( plugin_basename(__FILE__) );
			// unset activation notice
			unset( $_GET['activate'] );
			// display notice
			add_action( 'admin_notices', 'edd_slg_admin_notices' );
		}
	}
}
//Check Easy Digital Downloads plugin is Activated or not
add_action( 'admin_init', 'edd_slg_check_activation' );

/**
 * Activation Hook
 * Register plugin activation hook.
 *
 * @package Easy Digital Downloads - Social Login
 * @since 1.0.0
 */
register_activation_hook( __FILE__, 'edd_slg_install' );

/**
 * Plugin Setup (On Activation)
 *
 * Does the initial setup,
 * stest default values for the plugin options.
 *
 * @package Easy Digital Downloads - Social Login
 * @since 1.0.0
 */
function edd_slg_install() {

	// Plugin install setup function file
	require_once( EDD_SLG_DIR . '/includes/edd-slg-setup-functions.php' );

	// Manage plugin version wise settings when plugin install and activation
	edd_slg_manage_plugin_install_settings();
}

/**
 * Load Text Domain
 *
 * This gets the plugin ready for translation.
 *
 * @package Easy Digital Downloads - Social Login
 * @since 1.5.2
 */
function edd_slg_load_text_domain() {

	// Set filter for plugin's languages directory
	$edd_slg_lang_dir = dirname( plugin_basename(__FILE__) ) . '/languages/';
	$edd_slg_lang_dir = apply_filters( 'edd_slg_languages_directory', $edd_slg_lang_dir );

	// Traditional WordPress plugin locale filter
	$locale = apply_filters( 'plugin_locale', get_locale(), 'eddslg' );
	$mofile = sprintf( '%1$s-%2$s.mo', 'eddslg', $locale );

	// Setup paths to current locale file
	$mofile_local = $edd_slg_lang_dir . $mofile;
	$mofile_global = WP_LANG_DIR . '/' . EDD_SLG_BASENAME . '/' . $mofile;

	if( file_exists($mofile_global) ) { // Look in global /wp-content/languages/edd-social-login folder
		load_textdomain( 'eddslg', $mofile_global );
	} elseif( file_exists($mofile_local) ) { // Look in local /wp-content/plugins/edd-social-login/languages/ folder
		load_textdomain( 'eddslg', $mofile_local );
	} else { // Load the default language files
		load_plugin_textdomain( 'eddslg', false, $edd_slg_lang_dir );
	}
}

/**
 *
 * Handle to get url which are not used for script or functionality from plugin
 *
 * @package Easy Digital Downloads - Social Login
 * @since 2.4.6
 */
function edd_slg_get_blacklist() {

	$block_urls_list = array(
		'feed',
		'feed/rss',
		'feed/rss2',
		'feed/rdf',
		'feed/atom',
		'comments/feed'
	);

	// Look to see if WordPress is in a sub directory or this is a network site that uses sub directories
	$directory = str_replace( network_home_url(), '', get_site_url() );

	if( ! empty( $directory ) ) {
		foreach( $block_urls_list as $path ) {
			$block_urls_list[] = $directory . '/' . $path;
		}
	}

	return $block_urls_list;
}

/**
 * Handle to check should the session can start or not based on url
 *
 * @package Easy Digital Downloads - Social Login
 * @since 2.4.6
 */
function edd_slg_should_start_session() {

	$should_start_session = true;

	if( ! empty( $_SERVER[ 'REQUEST_URI' ] ) ) {

		$block_urls_list = edd_slg_get_blacklist();

		$uri       = ltrim( $_SERVER[ 'REQUEST_URI' ], '/' );
		$uri       = untrailingslashit( $uri );

		if( in_array( $uri, $block_urls_list ) ) {
			$should_start_session = false;
		}

		if( false !== strpos( $uri, 'feed=' ) ) {
			$should_start_session = false;
		}

		if( is_admin() && false !== strpos( $uri, 'wp-admin/admin-ajax.php' ) ) {
			// Do not want to start sessions in the admin unless we're processing an ajax request
			$should_start_session = false;
		}

		if( false !== strpos( $uri, 'wp_scrape_key' ) ) {
			// while saving the file editor save process
			$should_start_session = false;
		}

		if( false !== strpos( $uri, 'wp-json/wp-site-health/' ) ) {
            // Starting sessions while site health page gives error, so don't start
            $should_start_session = false;
        }

	}

	return $should_start_session;

}


//add action to load plugin
add_action('plugins_loaded', 'edd_slg_plugin_loaded');

function edd_slg_plugin_loaded() {

	//check easy digital downloads is activated or not
	if (class_exists('Easy_Digital_Downloads')) {

		// load first text domain
		edd_slg_load_text_domain();

		//check EDD_License class is exist
		if (class_exists('EDD_License')) {

			// Instantiate the licensing / updater. Must be placed in the main plugin file
			$license = new EDD_License(__FILE__, 'Social Login', EDD_SLG_VERSION, 'WPWeb');
		}

		/**
		 * Add plugin action links
		 *
		 * Adds a Settings, Docs link to the plugin list.
		 *
		 * @package Easy Digital Downloads - Social Login
		 * @since 1.5.1
		 */
		function edd_slg_add_plugin_links($links) {
			$plugin_links = array(
				'<a href="edit.php?post_type=download&page=edd-settings&tab=extensions">' . esc_html__('Settings', 'eddslg') . '</a>',
				'<a href="'.esc_url('https://docs.wpwebelite.com/edd-social-login/').'">' . esc_html__('Docs', 'eddslg') . '</a>'
			);

			return array_merge($plugin_links, $links);
		}
		add_filter('plugin_action_links_' . plugin_basename(__FILE__), 'edd_slg_add_plugin_links');

		/**
		 * Deactivation Hook
		 * Register plugin deactivation hook.
		 *
		 * @package Easy Digital Downloads - Social Login
		 * @since 1.0.0
		 */
		register_deactivation_hook(__FILE__, 'edd_slg_uninstall');

		/**
		 * Plugin Setup (On Deactivation)
		 * Delete  plugin options.
		 *
		 * @package Easy Digital Downloads - Social Login
		 * @since 1.0.0
		 */
		function edd_slg_uninstall() {

		}

		/**
		 * Start Session
		 * 
		 * @package Easy Digital Downloads - Social Login
		 * @since 1.0.0
		 */
		function edd_slg_start_session() {

			global $wpdb, $pagenow;

	    	// added code condition to fix wordpress site health rest api issue
	    	if( $pagenow != 'site-health.php' && (!isset($_GET['page']) || $_GET['page'] != 'health-check') ) {

	        	if( edd_slg_should_start_session() ) {

					if( !session_id() && !headers_sent() ) {
						session_start();
					}
				}
			}
		}

		//add action init for starting a session
		add_action('init', 'edd_slg_start_session');

		/**
		 * Includes Files
		 * 
		 * Includes some required files for plugin
		 *
		 * @package Easy Digital Downloads - Social Login
		 * @since 1.0.0
		 */
		global $edd_slg_model, $edd_slg_scripts,
		$edd_slg_render, $edd_slg_shortcodes,
		$edd_slg_public, $edd_slg_admin;

		if( !version_compare(PHP_VERSION, '5.4', '>=') ) {
			add_action( 'admin_notices', 'woo_slg_php_version' );
		} else {
			require_once(EDD_SLG_DIR . '/includes/WSL/Persistent/PersistentStorage.php');
		}

		// loads the Misc Functions file
		require_once( EDD_SLG_DIR . '/includes/edd-slg-misc-functions.php' );
		edd_slg_initialize();

		require_once(EDD_SLG_DIR . '/includes/class-edd-slg-persistant.php');
		$edd_slg_persistant_anonymous = new EddSocialLoginPersistentAnonymous();

		//social class loads
		require_once( EDD_SLG_SOCIAL_DIR . '/edd-slg-social.php');

		//Model Class for generic functions
		require_once( EDD_SLG_DIR . '/includes/class-edd-slg-model.php' );
		$edd_slg_model = new EDD_Slg_Model();

		//Scripts Class for scripts / styles
		require_once( EDD_SLG_DIR . '/includes/class-edd-slg-scripts.php' );
		$edd_slg_scripts = new EDD_Slg_Scripts();
		$edd_slg_scripts->add_hooks();

		//Renderer Class for HTML
		require_once( EDD_SLG_DIR . '/includes/class-edd-slg-renderer.php' );
		$edd_slg_render = new EDD_Slg_Renderer();

		//Shortcodes class for handling shortcodes
		require_once( EDD_SLG_DIR . '/includes/class-edd-slg-shortcodes.php' );
		$edd_slg_shortcodes = new EDD_Slg_Shortcodes();
		$edd_slg_shortcodes->add_hooks();

		//Public Class for public functionlities
		require_once( EDD_SLG_DIR . '/includes/class-edd-slg-public.php' );
		$edd_slg_public = new EDD_Slg_Public();
		$edd_slg_public->add_hooks();

		//Admin Pages Class for admin site
		require_once( EDD_SLG_ADMIN . '/class-edd-slg-admin.php' );
		$edd_slg_admin = new EDD_Slg_Admin();
		$edd_slg_admin->add_hooks();

		//Register Widget
		require_once( EDD_SLG_DIR . '/includes/widgets/class-edd-slg-login-buttons.php');

		//Loads the Templates Functions file
		require_once( EDD_SLG_DIR . '/includes/edd-slg-template-functions.php' );

		//Loads the Template Hook File
		require_once( EDD_SLG_DIR . '/includes/edd-slg-template-hooks.php' );

		//Loads the file to register block
		require_once( EDD_SLG_SOCIAL_BLOCK_DIR . '/social/index.php' );
	}
}

//Check WPWEB Updater is activated
if( class_exists('Wpweb_Upd_Admin') ) {
	
	// Plugin updates
	wpweb_queue_update( plugin_basename(__FILE__), EDD_SLG_PLUGIN_KEY );

	/**
	 * Include Auto Updating Files
	 * 
	 * @package Easy Digital Downloads - Social Login
	 * @since 1.0.0
	 */
	require_once( WPWEB_UPD_DIR . '/updates/class-plugin-update-checker.php' ); // auto updating
	$WpwebEddSlgUpdateChecker = new WpwebPluginUpdateChecker(
		WPWEB_UPD_DOMAIN . '/Updates/EDDSLG/license-info.php', __FILE__, EDD_SLG_PLUGIN_KEY
	);

	/**
	 * Auto Update
	 * Get the license key and add it to the update checker.
	 * 
	 * @package Easy Digital Downloads - Social Login
	 * @since 1.0.0
	 */
	function edd_slg_add_secret_key( $query ) {

		$plugin_key = EDD_SLG_PLUGIN_KEY;

		$query['lickey'] = wpweb_get_plugin_purchase_code($plugin_key);
		return $query;
	}

	$WpwebEddSlgUpdateChecker->addQueryArgFilter('edd_slg_add_secret_key');
} // end check WPWeb Updater is activated